// Slider functionality
let currentSlide = 0
const slides = document.querySelectorAll('.slide-fashion')
const dots = document.querySelectorAll('.dot-fashion')
const totalSlides = slides.length

function goToSlide(index) {
  if (slides.length === 0) return
  
  slides[currentSlide]?.classList.remove('active')
  dots[currentSlide]?.classList.remove('active')
  
  currentSlide = index
  if (currentSlide >= totalSlides) currentSlide = 0
  if (currentSlide < 0) currentSlide = totalSlides - 1
  
  slides[currentSlide]?.classList.add('active')
  dots[currentSlide]?.classList.add('active')
}

function goToNext() {
  goToSlide(currentSlide + 1)
}

function goToPrevious() {
  goToSlide(currentSlide - 1)
}

// Auto-slide
if (slides.length > 0) {
  setInterval(() => {
    goToNext()
  }, 3000)
  
  goToSlide(0)
}

// Initialize page
function initPage() {
  const currentPage = window.location.pathname.includes('enroll.html') ? 'enroll' : 'index'
  
  if (currentPage === 'enroll') {
    initEnrollPage()
  } else {
    initIndexPage()
  }
}

// Index page functions
function initIndexPage() {
  loadCourses()
  updateStats()
}

function loadCourses() {
  const saved = localStorage.getItem('sewingCourses')
  let courses = []
  
  if (saved) {
    try {
      courses = JSON.parse(saved)
    } catch (e) {
      courses = []
    }
  } else {
    courses = [
      {
        id: 1,
        project: 'Платье в стиле ретро',
        teacher: 'Елена Волкова',
        completionDate: '2025-01-15',
        resultPhoto: 'images/result-1.jpg'
      },
      {
        id: 2,
        project: 'Жилет из джинсовой ткани',
        teacher: 'Мария Смирнова',
        completionDate: '2024-12-20',
        resultPhoto: 'images/result-2.jpg'
      },
      {
        id: 3,
        project: 'Сумка-тоут с вышивкой',
        teacher: 'Анна Петрова',
        completionDate: '2024-11-10',
        resultPhoto: 'images/result-3.jpg'
      }
    ]
    localStorage.setItem('sewingCourses', JSON.stringify(courses))
  }
  
  renderCourses(courses)
}

function renderCourses(courses) {
  const container = document.getElementById('courses-grid')
  if (!container) return
  
  if (courses.length === 0) {
    container.innerHTML = `
      <div class="empty-fashion" style="grid-column: 1 / -1;">
        <div class="empty-icon-fashion">
          <img src="icons/scissors.svg" alt="Empty" style="width: 48px; height: 48px;">
        </div>
        <h2 class="empty-title-fashion">Нет проектов</h2>
        <p class="empty-text-fashion">Начните свой творческий путь с выбора курса</p>
        <button class="empty-button-fashion" onclick="window.location.href='enroll.html'">
          Записаться
        </button>
      </div>
    `
    return
  }
  
  container.innerHTML = courses.map(course => {
    const date = course.completionDate 
      ? new Date(course.completionDate).toLocaleDateString('ru-RU', { 
          day: 'numeric', 
          month: 'long', 
          year: 'numeric' 
        })
      : 'В процессе'
    
    return `
      <div class="course-card-fashion">
        <div class="course-image-fashion">
          <img src="${course.resultPhoto}" alt="${course.project}">
          <div class="course-status-fashion">Завершено</div>
        </div>
        <div class="course-content-fashion">
          <h3 class="course-title-fashion">${course.project}</h3>
          <div class="course-meta-fashion">
            <div class="meta-item-fashion">
              <img src="icons/user.svg" alt="User" style="width: 16px; height: 16px;">
              <span>${course.teacher}</span>
            </div>
            <div class="meta-item-fashion">
              <img src="icons/calendar.svg" alt="Calendar" style="width: 16px; height: 16px;">
              <span>${date}</span>
            </div>
          </div>
        </div>
      </div>
    `
  }).join('')
}

function updateStats() {
  const saved = localStorage.getItem('sewingCourses')
  const courses = saved ? JSON.parse(saved) : []
  
  const completed = courses.filter(c => c.completionDate).length
  const teachers = [...new Set(courses.map(c => c.teacher))].length
  
  document.getElementById('stat-projects')?.textContent = courses.length || 0
  document.getElementById('stat-teachers')?.textContent = teachers || 0
  document.getElementById('stat-completed')?.textContent = completed || 0
}

// Enroll page functions
function initEnrollPage() {
  renderLevels()
  setupForm()
}

const levels = [
  { 
    id: 'basic', 
    name: 'Базовый', 
    icon: 'scissors',
    color: 'var(--accent-coral)',
    description: 'Основы шитья для начинающих. Изучите базовые техники и создайте первые проекты.',
    features: ['Работа с тканью', 'Основные швы', 'Простая выкройка']
  },
  { 
    id: 'cutting', 
    name: 'Кройка', 
    icon: 'ruler',
    color: 'var(--accent-mint)',
    description: 'Профессиональная работа с выкройками. Научитесь создавать и корректировать лекала.',
    features: ['Снятие мерок', 'Построение выкройки', 'Корректировка']
  },
  { 
    id: 'modeling', 
    name: 'Моделирование', 
    icon: 'layers',
    color: 'var(--accent-lavender)',
    description: 'Создание сложных моделей одежды. Освойте техники моделирования и конструирования.',
    features: ['Моделирование', 'Конструирование', 'Сложные формы']
  },
  { 
    id: 'author', 
    name: 'Авторская одежда', 
    icon: 'palette',
    color: 'var(--charcoal)',
    description: 'Разработка уникальных дизайнов. Создавайте авторские коллекции и развивайте свой стиль.',
    features: ['Дизайн одежды', 'Авторские коллекции', 'Брендинг']
  }
]

function renderLevels() {
  const container = document.getElementById('levels-container')
  if (!container) return
  
  container.innerHTML = levels.map((level, index) => {
    return `
      <button 
        type="button" 
        class="level-item-fashion" 
        data-level="${level.id}"
        onclick="selectLevel('${level.id}')"
      >
        <div class="level-content-fashion">
          <div class="level-icon-fashion" style="background: ${level.color};">
            <img src="icons/${level.icon}.svg" alt="${level.name}" style="width: 28px; height: 28px;">
          </div>
          <div class="level-info-fashion">
            <h3 class="level-name-fashion">${level.name}</h3>
            <p class="level-desc-fashion">${level.description}</p>
            <div class="level-features-fashion">
              ${level.features.map(feature => `
                <span class="feature-badge-fashion" style="background: ${level.color}15; color: ${level.color};">
                  ${feature}
                </span>
              `).join('')}
            </div>
          </div>
        </div>
      </button>
    `
  }).join('')
}

let selectedLevel = ''
let selectedDate = ''

function selectLevel(id) {
  selectedLevel = id
  
  // Update UI
  document.querySelectorAll('.level-item-fashion').forEach(item => {
    const levelId = item.getAttribute('data-level')
    const level = levels.find(l => l.id === levelId)
    
    if (levelId === id) {
      item.classList.add('selected')
      item.style.borderColor = level.color
      
      // Add check indicator
      if (!item.querySelector('.check-mark-fashion')) {
        const check = document.createElement('div')
        check.className = 'check-mark-fashion'
        check.style.color = level.color
        check.innerHTML = `<img src="icons/check-circle.svg" alt="Check" style="width: 24px; height: 24px;">`
        item.appendChild(check)
      }
    } else {
      item.classList.remove('selected')
      item.style.borderColor = 'var(--border-clean)'
      
      const check = item.querySelector('.check-mark-fashion')
      if (check) check.remove()
    }
  })
  
  // Show date section
  const dateSection = document.getElementById('date-section')
  if (dateSection) {
    dateSection.style.display = 'block'
    dateSection.style.opacity = '1'
  }
  
  updateSubmitButton()
}

function setupForm() {
  const dateInput = document.getElementById('date-input')
  if (dateInput) {
    const today = new Date()
    const maxDate = new Date(today.getTime() + 180 * 24 * 60 * 60 * 1000)
    dateInput.min = today.toISOString().split('T')[0]
    dateInput.max = maxDate.toISOString().split('T')[0]
    
    dateInput.addEventListener('change', (e) => {
      selectedDate = e.target.value
      updateSubmitButton()
    })
  }
}

function updateSubmitButton() {
  const button = document.getElementById('submit-btn')
  if (button) {
    button.disabled = !selectedLevel || !selectedDate
  }
}

function handleSubmit(e) {
  if (e) e.preventDefault()
  
  if (!selectedLevel || !selectedDate) return
  
  const saved = localStorage.getItem('sewingCourses')
  const courses = saved ? JSON.parse(saved) : []
  
  const levelData = levels.find(l => l.id === selectedLevel)
  
  const newCourse = {
    id: Date.now(),
    project: `Проект: ${levelData.name}`,
    teacher: 'Преподаватель будет назначен',
    completionDate: null,
    resultPhoto: 'images/result-placeholder.jpg'
  }
  
  courses.push(newCourse)
  localStorage.setItem('sewingCourses', JSON.stringify(courses))
  
  // Show success
  const form = document.getElementById('enroll-form')
  if (form) {
    form.innerHTML = `
      <div class="success-fashion">
        <img src="icons/check-circle.svg" alt="Success" style="width: 32px; height: 32px;">
        <span>Запись оформлена! Переходим к вашим курсам...</span>
      </div>
    `
    
    setTimeout(() => {
      window.location.href = 'index.html'
    }, 2000)
  }
}

// Initialize on load
document.addEventListener('DOMContentLoaded', initPage)

